﻿using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace ImageToLayer2
{
    public partial class Form : System.Windows.Forms.Form
    {
        /// <summary>
        /// The bitmap image
        /// </summary>
        Bitmap image;

        /// <summary>
        /// Constructor
        /// </summary>
        public Form()
        {
            InitializeComponent();
        }

        /// <summary>
        /// When the exit menu item is clicked; Close the window
        /// </summary>
        /// <param name="sender">The object that sent the event</param>
        /// <param name="e">The event arguments</param>
        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }

        /// <summary>
        /// When the about menu item is clicked; Show the about box
        /// </summary>
        /// <param name="sender">The object that sent the event</param>
        /// <param name="e">The event arguments</param>
        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show(this, $"{Text}\r\n\r\n©2018 David Powell\r\nReleased into the public domain", Text, MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        /// <summary>
        /// When the open image menu item is clicked;
        /// Ask the user for a file and then open it
        /// </summary>
        /// <param name="sender">The object that sent the event</param>
        /// <param name="e">The event arguments</param>
        private void openImageToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                OpenFileDialog openFileDialog = new OpenFileDialog
                {
                    Filter = "Image files (*.bmp;*.gif;*.png)|*.bmp;*.gif;*.png|All files (*.*)|*.*",
                    CheckFileExists = true
                };

                if (openFileDialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                pictureBox.Image = null;
                image = null;
                saveRawBitmapToolStripMenuItem.Enabled = false;
                saveRaw9BitPaletteToolStripMenuItem.Enabled = false;

                image = (Bitmap)Image.FromFile(openFileDialog.FileName);

                if (image.PixelFormat != PixelFormat.Format8bppIndexed)
                {
                    image = null;
                    MessageBox.Show(this, "The image must be indexed 256 colours", Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                pictureBox.Image = image;

                saveRawBitmapToolStripMenuItem.Enabled = true;
                saveRaw9BitPaletteToolStripMenuItem.Enabled = true;
            }
            catch (Exception ex)
            {
                MessageBox.Show(this, $"Cannot open image file: {ex.Message}", Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// When the save raw bitmap menu item is clicked;
        /// Ask the user for a file and then save the raw data to it
        /// </summary>
        /// <param name="sender">The object that sent the event</param>
        /// <param name="e">The event arguments</param>
        private void saveRawBitmapToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                SaveFileDialog saveFileDialog = new SaveFileDialog
                {
                    Filter = "Raw image (*.raw)|*.raw|All files (*.*)|*.*",
                    AddExtension = true
                };

                if (saveFileDialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                Rectangle rect = new Rectangle(0, 0, image.Width, image.Height);
                BitmapData bitmapData = image.LockBits(rect, ImageLockMode.ReadOnly, PixelFormat.Format8bppIndexed);

                byte[] pixels = new byte[image.Width * image.Height];

                IntPtr bitmapPointer = bitmapData.Scan0;
                for (int y = 0; y < image.Height; y++)
                {
                    Marshal.Copy(bitmapPointer, pixels, image.Width * y, image.Width);
                    bitmapPointer += bitmapData.Stride;
                }

                Stream stream = saveFileDialog.OpenFile();
                stream.Write(pixels, 0, pixels.Length);
                stream.Close();

                MessageBox.Show(this, "Raw image saved successfully", Text, MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            catch (Exception ex)
            {
                MessageBox.Show(this, $"Cannot save the image: {ex.Message}", Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// When the save raw 9-bit palette menu item is clicked;
        /// Ask the user for a file and then save the raw palette data to it
        /// </summary>
        /// <param name="sender">The object that sent the event</param>
        /// <param name="e">The event arguments</param>
        private void saveRaw9BitPaletteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                SaveFileDialog saveFileDialog = new SaveFileDialog
                {
                    Filter = "Raw palette (*.pal)|*.pal|All files (*.*)|*.*",
                    AddExtension = true
                };

                if (saveFileDialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                byte[] palette = new byte[512];
                Color[] paletteColours = image.Palette.Entries;

                int offset = 0;
                foreach (Color colour in paletteColours)
                {
                    byte highByte = (byte)(colour.R & 0b11100000);
                    highByte |= (byte)((colour.G & 0b11100000) >> 3);
                    highByte |= (byte)((colour.B & 0b11000000) >> 6);

                    palette[offset++] = highByte;

                    byte lowByte = (byte)((colour.B & 0b00100000) >> 5);

                    palette[offset++] = lowByte;
                }

                Stream stream = saveFileDialog.OpenFile();
                stream.Write(palette, 0, palette.Length);
                stream.Close();

                MessageBox.Show(this, "Raw 9-Bit palette saved successfully", Text, MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            catch (Exception ex)
            {
                MessageBox.Show(this, $"Cannot save the palette: {ex.Message}", Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
    }
}
