; SD Card Load Routines
; EQU format
; V1.0
; Written by David Powell
; This file is released into the public domain


; Load a file into consecutive SRAM memory banks
; Switches memory at $E000 between 8K SRAM banks, restores original bank on success 
; In: IX = Address of NULL terminated file name
; In: BCDE = File offset
; In: HL = Number of bytes to load
; In: A = Starting SRAM bank
; Out: A = On error, ESXDOS error code
; Out: Carry Set = Error
; Corrupts: A, BC, DE, HL, IX
SD_LoadIntoBanks:
	; Store the parameters
	ld (@CurrentBank), a				; Store the starting SRAM bank number
	ld (@FileOffsetHigh), bc			; Store the high 16 bit word of the file offset
	ld (@FileOffsetLow), de				; Store the low 16 bit word of the file offset
	ld (@BytesLeft), hl					; Store the number of bytes to load
	
	; Remember the original SRAM bank currently at $E000
	ld a, NXRW_MEMORY_SLOT_7			; Select the memory slot 7 register ($E000)
	call NX_ReadRegister				; Read the register
	ld (@OriginalBank), a				; Store the original SRAM bank number

	; Open the file
	; IX already set
	call SD_OpenFile					; Open the file
	ret c								; If the carry flag is set, the file open failed, so return
	
	; Seek to the specified position in the file
	ld bc, (@FileOffsetHigh)			; Get the high 16 bit word of the file offset
	ld de, (@FileOffsetLow)				; Get the low 16 bit word of the file offset
	call SD_SeekFile					; Seek to the correct position in the file
	ret c								; If the carry flag is set, the file seek failed, so return

	ld ix, M_NX_MEMORY_SLOT_7			; Put destination address in IX
	
@nextblock:
	; Change to the next SRAM bank at $E000
	ld d, NXRW_MEMORY_SLOT_7			; Select the memory slot 7 register ($E000)
	ld a, (@CurrentBank)				; Get the current SRAM bank number
	call NX_WriteRegister				; Set the register
	
	ld hl, (@BytesLeft)					; Number of bytes left to load in HL
	ld bc, M_NX_MEMORY_SLOT_SIZE		; Bank size is 8K
	xor a								; clear carry
	sbc hl, bc							; Subtract bank size from remaining bytes
	jp nc, @fullsize					; jump if HL is positive as we are loading a full 8K bank
	ld bc, (@BytesLeft)					; load remaining bytes into BC as this is the final partial bank load
	ld hl, 0							; Set the remaining bytes to 0

@fullsize:
	ld (@BytesLeft), hl					; Store the remaining bytes

	; Read the next chunk of the file
	; IX set before the block loop
	call SD_ReadFile					; Read file chunk
	ret c								; Return if carry set, as there was an error
	
	; Increment current bank
	ld a, (@CurrentBank)				; Get the current SRAM bank number
	inc a								; Increment the SRAM bank number
	ld (@CurrentBank), a				; Store the new SRAM bank number
	
	; Check to see if there are any more bytes to load
	ld hl, (@BytesLeft)					; Read the remaining number of bytes left
	ld a, h								; Store high byte
	or l								; Compare high byte against low byte
	jp nz, @nextblock					; If high or low byte are not zero, there are more bytes left so loop again
	
	; Close the file
	call SD_CloseFile					; Close the file
	ret c								; Return if carry set, as there was an error
	
	; Restore original SRAM bank at $E000
	ld d, NXRW_MEMORY_SLOT_7			; Select memory slot 7 ($E000)
	ld a, (@OriginalBank)				; Select the original SRAM bank
	call NX_WriteRegister				; Set the register
	
	xor a								; Set result to 0 and clears carry flag
	;ld a, (@CurrentBank)				; Return the end bank, for debugging
	ret

; The current SRAM bank
@CurrentBank:
	db 0
	
; The original SRAM bank paged into memory slot 7 ($E000)
@OriginalBank:
	db 0
	
; The number of bytes left to load
@BytesLeft:
	dw 0
	
; The high 16 bit word of the file offset
@FileOffsetHigh:
	dw 0
	
; The low 16 bit word of the file offset
@FileOffsetLow:
	dw 0
	
; Open a file
; In: IX = Address of NULL terminated file name
; Corrupts: A, B
SD_OpenFile:
	ld a, ESXDOS_DRIVE_CURRENT							; Select the current drive
	ld b, ESXDOS_MODE_READ | ESXDOS_MODE_OPEN_EXIST		; Open an existing file for read-only
	rst ESXDOS_SYSCALL									; Call ESXDOS
    db ESXDOS_SYS_F_OPEN								; ESXDOS System File Open function
	ld (SD_FileHandle), a								; Store the file handle
	ret

; Read a file
; In: IX = Destination memory address
; In: BC = Number of bytes
; Corrupts: A
SD_ReadFile:
	ld a, (SD_FileHandle)								; Get the current file handle
    rst ESXDOS_SYSCALL									; Call ESXDOS
    db ESXDOS_SYS_F_READ								; ESXDOS System File Read function
    ret	

; Seek to the specified position in a file
; In: BCDE = File offset
; Corrupts: A, IX
SD_SeekFile:
	ld a, (SD_FileHandle)								; Get the current file handle
	ld ix, ESXDOS_SEEK_SET								; Set seek mode to absolute position
	rst ESXDOS_SYSCALL									; Call ESXDOS
	db ESXDOS_SYS_F_SEEK								; ESXDOS System File Seek function
	ret
	
; Close a file
; Corrupts: A
SD_CloseFile:
	ld a, (SD_FileHandle)								; Get the current file handler
    rst ESXDOS_SYSCALL									; Call ESXDOS
    db ESXDOS_SYS_F_CLOSE								; ESXDOS System File Close function
    ret

; The currently open file handle	
SD_FileHandle:
	db 0

