; Wibble - Interrupt mode 1 routine. Code to bank swap with the ROM
; EQU format
; V1.0
; Written by David Powell
; This file is released into the public domain

opt Z80
opt ZXNEXT

; Include the symbolic names
include "include/zxdefines.inc"
include "include/nextdefines.inc"

; Memory addresses to store variables
; As the Layer 2 screen will overlay the ROM area, we cannot write to addresses defined in this bank
; If we do, the Layer 2 screen will be modified/corrupted and the underlying memory will not be changed
; So instead we use known safe hardcoded addresses in normal RAM, just before the main code start address
MW_SINE_OFFSET equ    32996
MB_BORDER_COLOUR equ  32998
MB_LINE_INTERRUPT equ 32999

; Start at the top of address space, over the normal ROM area
pcorg $0000

start:
	ds $08								; Pad to RST $08
; RST $08
esxdos_syscall:

	ds $10								; Pad to RST $18
; RST $18
esxdos_romcall:

	ds $18								; Pad to RST $30
; RST $30
esxdos_auxcall:
	
	ds $8								; Pad to RST $38
; RST $38 Interrupt Mode 1 handler
rst38:
	ex af, af'							; Swap AF registers with shadow
	exx									; Swap BC, DE & HL with shadow
	
	; Uncommenting this section will alternate the border colour every time the interrupt is fired
	;ld a, (MB_BORDER_COLOUR)			; Load the border colour into A
	;xor ULA_BORDER_RED					; Flip colour on or off 
	;ld (MB_BORDER_COLOUR), a			; Load A into the border colour
	;out (P8RW_ULA), a					; Update the ULA with the border colour
	
	; Increment the line interrupt value and update the register
	ld d, NXRW_LINE_INTERRUPT_VALUE_LSB	; Select the line interrupt least significant byte
	ld a, (MB_LINE_INTERRUPT)			; Load the line interrupt into A
	inc a								; Increment the line interrupt
	ld (MB_LINE_INTERRUPT), a			; Write A back into the line interrupt
	call NX_WriteRegister				; Update the register
	
	; If we are not at line 0, at the top of the screen, skip the next frame setup
	or a								; Compare A against 0
	jp nz, not_at_top_of_screen			; Jump if line interrupt is not 0

	; When at the first line at the top of screen, setup the sine table offset for this frame
	
	; Get the offset address into the sine table for the last frame
	ld hl, sine							; Load the address of the start of the sine table into HL
	ld bc, (MW_SINE_OFFSET)				; Load the current sine offset into BC
	add hl, bc							; Add the offset to the sine table start to get the current sine table position
	
	; Reset the sine table offset to 0 if we get to position 256 as that is the repeat point
	ld a, c								; Load the low byte of the sine offset into A 
	or a								; Compare A against 0
	jp nz, dont_reset_sine				; Jump if low byte of the sine offset is not zero
	ld bc, 0							; Reset the sine offset
	
	; Increment the sine offset for this frame
dont_reset_sine:
	inc bc								; Increment the sine offset
	ld (MW_SINE_OFFSET), bc				; Store the sine offset
	
	; Perform the normal per-line routine
not_at_top_of_screen:

	; Set the Layer 2 X offset register to the current sine table offset value
	ld d, NXRW_LAYER2_OFFSET_X			; Select the layer 2 offset X register
	ld a, (hl)							; Load the sine value into A
	call NX_WriteRegister				; Update the register
	
	; Preserve the sine offset, then increment it by 7 for the Y offset
	push hl								; Preserve sine offset in HL
	ld de, 7							; Load DE with an additional sine offset of 7
	add hl, de							; Increase the sine offset in HL by 7
	
	; Set the Layer 2 Y offset register to the current sine table offset + 7 value / 2
	ld d, NXRW_LAYER2_OFFSET_Y			; Select the layer 2 offset Y register
	ld a, (hl)							; Load the sine value into A
	srl a								; Divide sine value by 2
	call NX_WriteRegister				; Update the register
	
	; Restore the X sine table offset
	pop hl								; Restore HL sine offset previously stored
	
	; Increment the sine offset, ready for the next line
	; Commenting out this line changes the effect to a pan, instead of a wibble
	inc hl								; Increment the sine offset in HL
	
	exx									; Swap AF registers with shadow
	ex af, af'							; Swap BC, DE & HL with shadow
	
	ei									; Enable interrupts
	ret									; Return from interrupt handler

; Sine table of 256 bytes in length, duplicated twice so that it can be offset	
sine:
	db $80,$83,$86,$89,$8c,$8f,$92,$95
	db $98,$9c,$9f,$a2,$a5,$a8,$ab,$ae
	db $b0,$b3,$b6,$b9,$bc,$bf,$c1,$c4
	db $c7,$c9,$cc,$ce,$d1,$d3,$d5,$d8
	db $da,$dc,$de,$e0,$e2,$e4,$e6,$e8
	db $ea,$eb,$ed,$ef,$f0,$f2,$f3,$f4
	db $f6,$f7,$f8,$f9,$fa,$fb,$fb,$fc
	db $fd,$fd,$fe,$fe,$fe,$ff,$ff,$ff
	db $ff,$ff,$ff,$ff,$fe,$fe,$fd,$fd
	db $fc,$fc,$fb,$fa,$f9,$f8,$f7,$f6
	db $f5,$f4,$f2,$f1,$ef,$ee,$ec,$eb
	db $e9,$e7,$e5,$e3,$e1,$df,$dd,$db
	db $d9,$d7,$d4,$d2,$cf,$cd,$ca,$c8
	db $c5,$c3,$c0,$bd,$ba,$b8,$b5,$b2
	db $af,$ac,$a9,$a6,$a3,$a0,$9d,$9a
	db $97,$94,$91,$8e,$8a,$87,$84,$81
	db $7e,$7b,$78,$75,$71,$6e,$6b,$68
	db $65,$62,$5f,$5c,$59,$56,$53,$50
	db $4d,$4a,$47,$45,$42,$3f,$3c,$3a
	db $37,$35,$32,$30,$2d,$2b,$28,$26
	db $24,$22,$20,$1e,$1c,$1a,$18,$16
	db $14,$13,$11,$10,$0e,$0d,$0b,$0a
	db $09,$08,$07,$06,$05,$04,$03,$03
	db $02,$02,$01,$01,$00,$00,$00,$00
	db $00,$00,$00,$01,$01,$01,$02,$02
	db $03,$04,$04,$05,$06,$07,$08,$09
	db $0b,$0c,$0d,$0f,$10,$12,$14,$15
	db $17,$19,$1b,$1d,$1f,$21,$23,$25
	db $27,$2a,$2c,$2e,$31,$33,$36,$38
	db $3b,$3e,$40,$43,$46,$49,$4c,$4f
	db $51,$54,$57,$5a,$5d,$60,$63,$67
	db $6a,$6d,$70,$73,$76,$79,$7c,$80
	db $80,$83,$86,$89,$8c,$8f,$92,$95
	db $98,$9c,$9f,$a2,$a5,$a8,$ab,$ae
	db $b0,$b3,$b6,$b9,$bc,$bf,$c1,$c4
	db $c7,$c9,$cc,$ce,$d1,$d3,$d5,$d8
	db $da,$dc,$de,$e0,$e2,$e4,$e6,$e8
	db $ea,$eb,$ed,$ef,$f0,$f2,$f3,$f4
	db $f6,$f7,$f8,$f9,$fa,$fb,$fb,$fc
	db $fd,$fd,$fe,$fe,$fe,$ff,$ff,$ff
	db $ff,$ff,$ff,$ff,$fe,$fe,$fd,$fd
	db $fc,$fc,$fb,$fa,$f9,$f8,$f7,$f6
	db $f5,$f4,$f2,$f1,$ef,$ee,$ec,$eb
	db $e9,$e7,$e5,$e3,$e1,$df,$dd,$db
	db $d9,$d7,$d4,$d2,$cf,$cd,$ca,$c8
	db $c5,$c3,$c0,$bd,$ba,$b8,$b5,$b2
	db $af,$ac,$a9,$a6,$a3,$a0,$9d,$9a
	db $97,$94,$91,$8e,$8a,$87,$84,$81
	db $7e,$7b,$78,$75,$71,$6e,$6b,$68
	db $65,$62,$5f,$5c,$59,$56,$53,$50
	db $4d,$4a,$47,$45,$42,$3f,$3c,$3a
	db $37,$35,$32,$30,$2d,$2b,$28,$26
	db $24,$22,$20,$1e,$1c,$1a,$18,$16
	db $14,$13,$11,$10,$0e,$0d,$0b,$0a
	db $09,$08,$07,$06,$05,$04,$03,$03
	db $02,$02,$01,$01,$00,$00,$00,$00
	db $00,$00,$00,$01,$01,$01,$02,$02
	db $03,$04,$04,$05,$06,$07,$08,$09
	db $0b,$0c,$0d,$0f,$10,$12,$14,$15
	db $17,$19,$1b,$1d,$1f,$21,$23,$25
	db $27,$2a,$2c,$2e,$31,$33,$36,$38
	db $3b,$3e,$40,$43,$46,$49,$4c,$4f
	db $51,$54,$57,$5a,$5d,$60,$63,$67
	db $6a,$6d,$70,$73,$76,$79,$7c,$80

; Include the library routines
include "library/nextlib.asm"
