/*
  bugsquish.c
  
  Bug Squish

  by Bill Kendrick
  bill@newbreedsoftware.com
  http://www.newbreedsoftware.com/
  
  March 19, 2000 - March 24, 2000
*/


#define VERSION "0.0.1"


/* #ncludes: */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <SDL/SDL.h>

#ifndef NOSOUND
#include <SDL/SDL_mixer.h>
#endif

#include "data/images/flyswatter.xbm"
#include "data/images/flyswatter-mask.xbm"


#define POINTS_FOR_BONUS_BLOOD 10000 /* How many points til bonus blood */
#define MAX_BUGS_TO_START 8          /* When beginning level, limit to this */
#define CHANCE_OF_EXTRA 200          /* How often to add extras */
#define CHANCE_OF_OUCH 20            /* How often, when being sucked, to say */
#define CHANCE_OF_FAST_BUG 5         /* How often, when adding bugs, fast */
#define CHANCE_OF_SPORATICNESS 20    /* How often to change direction */


/* Image enumerations: */

enum {
  IMG_PRESENTS,
  IMG_TITLE,
  IMG_ARM1,
  IMG_ARM2,
  IMG_ARM3,
  IMG_ARM4,
  IMG_ARM5,
  IMG_GAUGE_FULL,
  IMG_GAUGE_EMPTY,
  IMG_SCOREBOX,
  IMG_HIGHSCORE,
  IMG_WAVEBOX,
  IMG_BONUS,
  IMG_PAUSED,
  IMG_NUMBERS,
  IMG_LARGE0,
  IMG_LARGE1,
  IMG_LARGE2,
  IMG_LARGE3,
  IMG_LARGE4,
  IMG_LARGE5,
  IMG_LARGE6,
  IMG_LARGE7,
  IMG_LARGE8,
  IMG_LARGE9,
  IMG_BUG0A,
  IMG_BUG0B,
  IMG_BUG0C,
  IMG_BUG1A,
  IMG_BUG1B,
  IMG_BUG1C,
  IMG_BUG2A,
  IMG_BUG2B,
  IMG_BUG2C,
  IMG_BUG3A,
  IMG_BUG3B,
  IMG_BUG3C,
  IMG_BLOOD,
  IMG_BLOOD_SQUISHED,
  IMG_MULTIPLIER,
  IMG_MULTIPLIER_SQUISHED,
  IMG_TIMES2,
  NUM_IMAGES
};


/* Image filenames: */

const char * image_names [NUM_IMAGES] = {
   "images/presents.bmp",
   "images/title.bmp",
   "images/arm1.bmp",
   "images/arm2.bmp",
   "images/arm3.bmp",
   "images/arm4.bmp",
   "images/arm5.bmp",
   "images/gauge-full.bmp",
   "images/gauge-empty.bmp",
   "images/scorebox.bmp",
   "images/highscore.bmp",
   "images/wavebox.bmp",
   "images/bonus.bmp",
   "images/paused.bmp",
   "images/numbers.bmp",
   "images/large0.bmp",
   "images/large1.bmp",
   "images/large2.bmp",
   "images/large3.bmp",
   "images/large4.bmp",
   "images/large5.bmp",
   "images/large6.bmp",
   "images/large7.bmp",
   "images/large8.bmp",
   "images/large9.bmp",
   "images/bug0a.bmp",
   "images/bug0b.bmp",
   "images/bug0c.bmp",
   "images/bug1a.bmp",
   "images/bug1b.bmp",
   "images/bug1c.bmp",
   "images/bug2a.bmp",
   "images/bug2b.bmp",
   "images/bug2c.bmp",
   "images/bug3a.bmp",
   "images/bug3b.bmp",
   "images/bug3c.bmp",
   "images/blood.bmp",
   "images/blood-squished.bmp",
   "images/mult.bmp",
   "images/mult-squished.bmp",
   "images/times2.bmp"
};


/* Bug enumerations: */

enum {
  BUG_MOSQUITO,
  BUG_TICK,
  BUG_ROACH,
  BUG_STICK,
  NUM_BUGS
};


/* Sound enumerations: */

enum {
  SND_SQUISH1,
  SND_SQUISH2,
  SND_OUCH,
  SND_UGH,
  SND_GLUG,
  SND_BONUS,
  SND_AH,
  SND_HIGHSCORE,
  NUM_SOUNDS
};


/* Sound filenames: */

const char * sound_names[NUM_SOUNDS] = {
   "sounds/squish2.wav",
   "sounds/squish1.wav",
   "sounds/ouch.wav",
   "sounds/ugh.wav",
   "sounds/glug.wav",
   "sounds/bonus.wav",
   "sounds/ah.wav",
   "sounds/highscore.wav"
};


/* Music filenames: */

const char* MUS_TITLE="music/corpses.mod";
const char* MUS_GAME="music/adventures.mod";


/* Bug type: */

typedef struct bug_type {
  int alive, kind, x, y, xm, ym, xmm, target_y;
} bug_type;

typedef struct splat_type {
  int active, img, x, y;
} splat_type;

typedef struct extra_type {
  int active, kind, x, y, xm, ym;
} extra_type;


#define MAX_BUGS 64
#define MAX_SPLATS 256


/* Local function prototypes: */

void setup(void);
int title(void);
int game(void);
void seticon(void);
int pause_screen(void);
void playsound(int snd, int chan);
void resetlevel(void);
void addbug(void);
void addsplat(int x, int y, int kind);
void intro(void);
void drawbonusnumbers(int num);
void incrementscore(int amt);
void addblood(void);
void drawdigits(int v, int x, int y);
void loaddata(void);
void savedata(void);
FILE * opendata(char * mode);
void getargs(int argc, char * argv[]);
void usage(int ret);
char* datafilename(const char* name);

/* Global variables: */

SDL_Surface * screen;
SDL_Surface * images[NUM_IMAGES];
int use_fullscreen, use_sound;
int level, score, highscore, highlevel, blood, bugs_added, multiplier,
  has_highscore;
bug_type bugs[MAX_BUGS];
splat_type splats[MAX_SPLATS];
extra_type extra;
char datapath[1024];
char datafile[2048];

#ifndef NOSOUND
Mix_Chunk * sounds[NUM_SOUNDS];
Mix_Music * mus_title, * mus_game;
#endif


/* --- MAIN --- */

int main(int argc, char * argv[])
{
  int done;
  int i;
  
  
  /* Check for arguments: */
  
  getargs(argc, argv);


  /* get data directory path */
  
  for (i=strlen(argv[0]) ; argv[0][i]!='/' ; i--);
  strncpy(datapath, argv[0], i);  
  strcat(datapath,"/data/");


  /* Setup: */
    
  setup();
  
    
  /* Load data: */
  
  loaddata();
  
  
  /* Intro screen: */
  
  intro();
  
  
  /* MAIN FUNCTION LOOP: */
  
  done = 0;
  
  do
    {
      done = title();
      
      if (!done)
	done = game();
    }
  while (!done);
  
  
  /* Save data: */
  
  savedata();
  
  
  /* Quit and exit: */
  
  SDL_Quit();
  
  return(0);
}


/* Game function! */

int game(void)
{
  SDL_Rect src, dest;
  SDL_Event event;
  int done, quit, gameover, i, frame, num_bugs, next_level, old_blood, bonus;
  
  
  /* Start the game! */
  
  level = 1;
  score = 0;
  has_highscore = 0;
  multiplier = 1;
  blood = (images[IMG_GAUGE_FULL] -> h);
  old_blood = 0;
  bonus = 0;
  next_level = 0;
  
  resetlevel();
  
  
  /* MAIN GAME LOOP! */
  
  done = 0;
  gameover = 0;
  quit = 0;
  frame = 0;
    
  do
    {
      frame++;
      
      /* Handle events: */
      
      while (SDL_PollEvent(&event) > 0)
	{
	  if (event.type == SDL_QUIT)
	    {
	      /* WM Quit - Quit! */
	      
	      quit = 1;
	    }
	  else if (event.type == SDL_KEYDOWN)
	    {
	      /* Keypress! ESCAPE - Quit, Space - Pause!*/
	      
	      if (event.key.keysym.sym == SDLK_ESCAPE)
		done = 1;
	      else if (event.key.keysym.sym == SDLK_SPACE)
		{
		  /* Pause! */
		  
		  quit = pause_screen();
		}
	    }
	  else if (event.type == SDL_MOUSEBUTTONDOWN)
	    {
	      if (!gameover)
		{
		  /* Mouse click! - Squish! */
		  
		  for (i = 0; i < MAX_BUGS; i++)
		    {
		      if (bugs[i].alive &&
			  event.button.x >= bugs[i].x - 16 &&
			  event.button.x <= bugs[i].x + 32 &&
			  event.button.y >= bugs[i].y - 16 &&
			  event.button.y <= bugs[i].y + 32)
			{
			  bugs[i].alive = 0;
			  playsound(SND_SQUISH1 + (rand() % 2), 0);
			  addsplat(bugs[i].x, bugs[i].y,
				   (IMG_BUG0C + bugs[i].kind * 3));
			  incrementscore((480 - bugs[i].y) / 4);
			}
		    }
		  
		  
		  /* Grab an extra? */
		  
		  if (extra.active &&
		      event.button.x >= extra.x - 16 &&
		      event.button.x <= extra.x + 32 &&
		      event.button.y >= extra.y - 16 &&
		      event.button.y <= extra.y + 32)
		    {
		      if (extra.kind == 0)
			{
			  /* Extra blood! */
			  
			  addblood();
			}
		      else
			{
			  /* Multiplier! */
			  
			  if (multiplier == 1)
			    {
			      multiplier = 2;
			      playsound(SND_BONUS, 1);
			    }
			  else
			    playsound(SND_GLUG, 1);
			}
		      
		      addsplat(extra.x, extra.y,
			       IMG_BLOOD_SQUISHED + extra.kind * 2);
		      extra.active = 0;
		    }
		}
	    }
	}
      
      
      /* Move all bugs: */
      
      num_bugs = 0;
      
      for (i = 0; i < MAX_BUGS; i++)
	{
	  if (bugs[i].alive)
	    {
	      num_bugs++;
	      
	      /* Move: */
	      
	      bugs[i].x = bugs[i].x + bugs[i].xm;
	      bugs[i].y = bugs[i].y + bugs[i].ym;
	      
	      
	      if (bugs[i].kind == BUG_MOSQUITO)
		{
		  /* Wave back and forth? */
		  
		  bugs[i].xm = bugs[i].xm + bugs[i].xmm;
		  
		  if (bugs[i].xm < -8)
		    bugs[i].xmm = 1;
		  else if (bugs[i].xm > 8)
		    bugs[i].xmm = -1;
		}
	      else if (bugs[i].kind == BUG_STICK)
		{
		  /* Move horizontally sporatically? */
		  
		  if (rand() % CHANCE_OF_SPORATICNESS)
		    bugs[i].xm = (rand() % (level * 2)) - level;
		}
	      else if (bugs[i].kind == BUG_ROACH)
		{
		  /* Move vertically sporatically? */
		  
		  if (rand() % CHANCE_OF_SPORATICNESS)
		    bugs[i].ym = (rand() % (level * 2)) - (level / 2);
		}
	      
	      
	      /* Land? */
	      
	      if (bugs[i].y >= bugs[i].target_y && bugs[i].ym > 0)
		{
		  bugs[i].alive = bugs[i].target_y;
		  bugs[i].xm = 0;
		  bugs[i].xmm = 0;
		  bugs[i].ym = 0;
		  
		  playsound(SND_OUCH, 2);
		}
	      
	      
	      /* Fly away? */
	      
	      if (gameover)
		{
		  if (bugs[i].ym >= 0)
		    bugs[i].ym = -2;
		  
		  if (bugs[i].y <= -32)
		    bugs[i].alive = 0;
		}
	      
	      
	      /* Keep in horizontal bounds: */
	      
	      if (bugs[i].x < images[IMG_GAUGE_FULL] -> w)
		{
		  bugs[i].x = images[IMG_GAUGE_FULL] -> w;
		  bugs[i].xm = 1;
		}
	      else if (bugs[i].x > 607)
		{
		  bugs[i].x = 607;
		  bugs[i].xm = -1;
		}
	      
	      
	      /* Suck blood! */
	      
	      if (!gameover && bugs[i].y >= bugs[i].target_y)
		{
		  if ((frame % 4) == 0)
		    {
		      blood--;
		      
#ifndef NOSOUND
		      if (use_sound)
			{
			  if ((rand() % CHANCE_OF_OUCH) == 0 && !gameover &&
			      !Mix_Playing(2))
			    {
			      playsound(SND_OUCH, 2);
			    }
			}
#endif
		      
		      if (blood <= 0)
			{
			  blood = 0;
			  
			  if (!gameover)
			    {
			      gameover = 1;
			      playsound(SND_UGH, 2);
			      
			      
			      /* Make bugs fly away (all done! yum!) */
			      
			      for (i = 0; i < MAX_BUGS; i++)
				bugs[i].ym = -(rand() % 5) - 3;
			    }
			}
		    }
		}
	    }
	}


      /* Handle extra: */
      
      if (extra.active)
	{
	  /* Move it: */
	  
	  extra.x = extra.x + extra.xm;
	  extra.y = extra.y + extra.ym;
	  
	  
	  /* Out of bounds? */
	  
	  if (extra.x < -32 || extra.x > 640 ||
	      extra.y < -32 || extra.y > 480)
	    {
	      extra.active = 0;
	    }
	}
      else if ((rand() % CHANCE_OF_EXTRA) == 0 && !next_level && !gameover)
	{
	  /* Add an extra! */
	  
	  extra.active = 1;
	  extra.kind = (rand() % 2);
	  
	  if ((rand() % 2) == 0)
	    {
	      /* From the top! */
	      
	      extra.y = -32;
	      extra.x = (rand() % 608);
	      
	      extra.xm = (rand() % 32) - 16;
	      extra.ym = (rand() % 16);
	    }
	  else
	    {
	      /* From one of the sides! */
	      
	      if ((rand() % 2) == 0)
		{
		  /* Left: */
		  
		  extra.x = -32;
		  extra.xm = (rand() % 16);
		}
	      else
		{
		  /* Left: */
		  
		  extra.x = 640;
		  extra.xm = -(rand() % 16);
		}
	      
	      extra.ym = (rand() % 32) - 16;
	    }
	}
      
      
      /* Quit if game's over! */
      
      if (num_bugs == 0 && gameover)
	done = 1;
      
      
      /* Erase screen: */
      
      SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 255, 255, 255));
      
      
      /* Draw arm: */
      
      dest.x = 0;
      dest.y = 480 - images[IMG_ARM1] -> h;
      dest.w = 640;
      dest.h = images[IMG_ARM1] -> h;
      
      if (!next_level)
	{
	  /* Faded based on blood level: */
	  
	  SDL_BlitSurface(images[(IMG_ARM5 -
				  ((blood * 5) /
				   ((images[IMG_GAUGE_FULL] -> h) + 1)))],
			  NULL, screen, &dest);
	}
      else
	{
	  /* Blood bonus happening - show based on recorded blood level: */
	  
	  SDL_BlitSurface(images[(IMG_ARM5 -
				  ((old_blood * 5) /
				   ((images[IMG_GAUGE_FULL] -> h) + 1)))],
			  NULL, screen, &dest);
	}
      

      /* Draw all splats: */

      for (i = 0; i < MAX_SPLATS; i++)
	{
	  if (splats[i].active)
	    {
	      dest.x = splats[i].x;
	      dest.y = splats[i].y;
	      dest.w = 32;
	      dest.h = 32;
	      
	      SDL_BlitSurface(images[splats[i].img], NULL,
			      screen, &dest);
	    }
	}
      
      
      /* Draw all bugs: */

      for (i = 0; i < MAX_BUGS; i++)
	{
	  if (bugs[i].alive)
	    {
	      dest.x = bugs[i].x;
	      dest.y = bugs[i].y;
	      dest.w = 32;
	      dest.h = 32;
	      
	      if (bugs[i].ym != 0)
		SDL_BlitSurface(images[IMG_BUG0A + bugs[i].kind * 3 +
				      ((frame + i) % 2)], NULL,
				screen, &dest);
	      else
		SDL_BlitSurface(images[IMG_BUG0A + bugs[i].kind * 3 +
				      ((frame / 8) % 2)], NULL,
				screen, &dest);
	    }
	}

      
      /* Draw extra: */
      
      if (extra.active)
	{
	  dest.x = extra.x;
	  dest.y = extra.y;
	  dest.w = 32;
	  dest.h = 32;
	  
	  SDL_BlitSurface(images[IMG_BLOOD + extra.kind * 2], NULL,
			  screen, &dest);
	}
      
      
      /* Draw blood gauge: */
      
      src.x = 0;
      src.y = (images[IMG_GAUGE_FULL] -> h) - blood;
      src.w = (images[IMG_GAUGE_FULL] -> w);
      src.h = blood;
      
      dest.x = src.x;
      dest.y = src.y;
      dest.w = src.w;
      dest.h = src.h;
      
      SDL_BlitSurface(images[IMG_GAUGE_FULL], &src, screen, &dest);
      
      src.x = 0;
      src.y = 0;
      src.w = (images[IMG_GAUGE_EMPTY] -> w);
      src.h = (images[IMG_GAUGE_EMPTY] -> h) - blood;
      
      dest.x = src.x;
      dest.y = src.y;
      dest.w = src.w;
      dest.h = src.h;
      
      SDL_BlitSurface(images[IMG_GAUGE_EMPTY], &src, screen, &dest);
      
      
      /* Multiplier value: */

      if (multiplier == 2)
	{
	  dest.x = ((images[IMG_GAUGE_FULL] -> w) / 2) - 8;
	  dest.y = (images[IMG_GAUGE_FULL] -> h) + 2;
	  dest.w = 16;
	  dest.h = 16;
	  
	  SDL_BlitSurface(images[IMG_TIMES2], NULL, screen, &dest);
	}
      
      
      /* Score: */
      
      dest.x = 0;
      dest.y = 440;
      dest.w = 128;
      dest.h = 40;
      
      SDL_BlitSurface(images[IMG_SCOREBOX], NULL, screen, &dest);
      
      drawdigits(score, 2, 462);
      
      
      /* High score bubble? */
      
      if (has_highscore)
	{
	  dest.x = 110;
	  dest.y = 462;
	  dest.w = 16;
	  dest.h = 16;
	  
	  SDL_BlitSurface(images[IMG_HIGHSCORE], NULL, screen, &dest);
	}


      /* Wave: */
      
      dest.x = 576;
      dest.y = 440;
      dest.w = 64;
      dest.h = 40;
      
      SDL_BlitSurface(images[IMG_WAVEBOX], NULL, screen, &dest);
      
      drawdigits(level, 578, 462);
      
      
      /* Go to next level? */
      
      if (num_bugs == 0 && bugs_added >= level * 4 && !gameover)
	{
	  level++;
	  
	  if (level > 32)
	    level = 32;
	  
	  next_level = 1;
	  old_blood = blood;
	  bonus = 0;
	}
      
      
      /* Add more bugs: */
      
      if (((rand() % (35 - level)) == 0 || num_bugs == 0)
	  && bugs_added < level * 4 && !next_level && !gameover)
	addbug();
      
      
      if (next_level == 1)
	{
	  /* Grab bonus for blood: */
	  
	  if (blood > 5)
	    {
	      bonus = bonus + 5 * multiplier;
	      blood = blood - 5;
	    }
	  else
	    {
	      blood = 0;
	      bonus = bonus + blood * multiplier;
	    }

	  
#ifndef NOSOUND
	  if (use_sound)
	    {
	      if (!Mix_Playing(1))
		{
		  playsound(SND_GLUG, 1);
		}
	    }
#endif
	  
	  
	  /* Draw bonus box: */
	  
	  dest.x = 192;
	  dest.y = 176;
	  dest.w = 256;
	  dest.h = 128;
	  
	  SDL_BlitSurface(images[IMG_BONUS], NULL,
			  screen, &dest);
	  
	  drawbonusnumbers(bonus);
	  
	  
	  /* No more blood - go to next level! */
	  
	  if (blood <= 0)
	    {
	      blood = old_blood;
	      next_level = 0;
	      
	      resetlevel();
	      playsound(SND_BONUS, 1);
	      
	      incrementscore(bonus);
	    }
	}
      
      
      /* Update screen, play music, keep framerate stable: */
      
      SDL_Flip(screen);
      SDL_Delay(30);
      
#ifndef NOSOUND
      if (use_sound)
	{
	  if (!Mix_PlayingMusic())
	    {
	      Mix_PlayMusic(mus_game, 0);
	    }
	}
#endif
    }
  while (!done && !quit);
  

#ifndef NOSOUND
  if (use_sound)
    Mix_HaltMusic();
#endif

  return(quit);
}


/* Setup function: */

void setup(void)
{
  SDL_Surface * image;
  int i;
  SDL_Rect dest;
  Uint8 temp_bitmap[32], temp_bitmask[32];
  Uint8 b;

  
  /* Init SDL Video: */
  
  if (SDL_Init(SDL_INIT_VIDEO) < 0)
    {
      fprintf(stderr,
              "\nError: I could not initialize video!\n"
              "The Simple DirectMedia error that occured was:\n"
              "%s\n\n", SDL_GetError());
      exit(1);
    }

  
  /* Init SDL Audio: */
  
  if (use_sound == 1)
    {
      if (SDL_Init(SDL_INIT_AUDIO) < 0)
        {
          fprintf(stderr,
                  "\nWarning: I could not initialize audio!\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", SDL_GetError());
          use_sound = 0;
        }
    }
  
  
  /* Open sound: */
  
#ifndef NOSOUND
  if (use_sound == 1)
    {
      if (Mix_OpenAudio(22050, AUDIO_S16, 2, 256) < 0)
        {
          fprintf(stderr,
                  "\nWarning: I could not set up audio for 22050 Hz "
                  "16-bit stereo.\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", SDL_GetError());
          use_sound = 0;
        }
    }
#endif


  /* Open display: */
  
  if (use_fullscreen == 1)
    {
      screen = SDL_SetVideoMode(640, 480, 16, SDL_FULLSCREEN | SDL_HWSURFACE);
      if (screen == NULL)
        {
          fprintf(stderr,
                  "\nWarning: I could not set up fullscreen video for "
                  "640x480 mode.\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", SDL_GetError());
          use_fullscreen = 0;
        }
    }
  
  if (use_fullscreen == 0)
    {
      screen = SDL_SetVideoMode(640, 480, 16, 0);
      
      if (screen == NULL)
        {
          fprintf(stderr,
                  "\nError: I could not set up video for 640x480 mode.\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", SDL_GetError());
          exit(1);
        }
    }
  
  
  /* Set icon image: */
  
  seticon();
  
  
  /* Set window manager stuff: */
  
  SDL_WM_SetCaption("Bug Squish", "Bug Squish");


  /* Make cursor: */
  
  for (i = 0; i < 32; i++)
    {
      b = flyswatter_bits[i];
      
      temp_bitmap[i] = (((b & 0x01) << 7) |
			((b & 0x02) << 5) |
			((b & 0x04) << 3) |
			((b & 0x08) << 1) |
			((b & 0x10) >> 1) |
			((b & 0x20) >> 3) |
			((b & 0x40) >> 5) |
			((b & 0x80) >> 7));
      
      b = flyswatter_mask_bits[i];
      
      temp_bitmask[i] = (((b & 0x01) << 7) |
			 ((b & 0x02) << 5) |
			 ((b & 0x04) << 3) |
			 ((b & 0x08) << 1) |
			 ((b & 0x10) >> 1) |
			 ((b & 0x20) >> 3) |
			 ((b & 0x40) >> 5) |
			 ((b & 0x80) >> 7));
    }
  
  SDL_SetCursor(SDL_CreateCursor(temp_bitmap, temp_bitmask, 16, 16, 0, 0));
  
  
  /* Load graphics: */
  
  for (i = 0; i < NUM_IMAGES; i++)
    {
      /* Load image file: */
      
      image = SDL_LoadBMP(datafilename(image_names[i]));

      if (image == NULL)
        {
          fprintf(stderr,
                  "\nError: I couldn't load a graphics file:\n"
                  "%s\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", datafilename(image_names[i]), SDL_GetError());
          exit(1);
        }
      
      
      /* Set transparency: */
      
      images[i] = SDL_DisplayFormat(image);
      if (images[i] == NULL)
        {
          fprintf(stderr,
                  "\nError: I couldn't convert a file to the display format:\n"
                  "%s\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", datafilename(image_names[i]), SDL_GetError());
          exit(1);
        }
      
      if (SDL_SetColorKey(images[i], (SDL_SRCCOLORKEY),
                          SDL_MapRGB(images[i] -> format,
                                     0xFF, 0xFF, 0xFF)) == -1)
        {
          fprintf(stderr,
                  "\nError: I could not set the color key for the file:\n"
                  "%s\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", datafilename(image_names[i]), SDL_GetError());
          exit(1);
        }
      
      SDL_FreeSurface(image);
      
      
      /* Draw percentage bar: */
      
      dest.x = 0;
      dest.y = 470;
      dest.w = (640 * i) / NUM_IMAGES;
      dest.h = 10;
      
      SDL_FillRect(screen, &dest, SDL_MapRGB(screen->format, 255, 255, 255));
      SDL_UpdateRect(screen, dest.x, dest.y, dest.w, dest.h);
      SDL_Delay(1);
    }
  
  
#ifndef NOSOUND
  if (use_sound == 1)
    {
      /* Load sounds: */
      
      for (i = 0; i < NUM_SOUNDS; i++)
        {
          sounds[i] = Mix_LoadWAV(datafilename(sound_names[i]));
          if (sounds[i] == NULL)
            {
              fprintf(stderr,
                      "\nError: I could not load the sound file:\n"
                      "%s\n"
                      "The Simple DirectMedia error that occured was:\n"
                      "%s\n\n", datafilename(sound_names[i]), SDL_GetError());
              exit(1);
            }
        }
      
      
      /* Load musics: */
      
      /* (title) */
      
      mus_title = Mix_LoadMUS(datafilename(MUS_TITLE));
      if (mus_title == NULL)
        {
          fprintf(stderr,
                  "\nError: I could not load the music file:\n"
                  "%s\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", datafilename(MUS_TITLE), SDL_GetError());
          exit(1);
        }

      /* (game) */
      
      mus_game = Mix_LoadMUS(datafilename(MUS_GAME));
      if (mus_game == NULL)
        {
          fprintf(stderr,
                  "\nError: I could not load the music file:\n"
                  "%s\n"
                  "The Simple DirectMedia error that occured was:\n"
                  "%s\n\n", datafilename(MUS_GAME), SDL_GetError());
          exit(1);
        }
      /* (gameover) */
    }
#endif

  
  /* Seed random generator: */
  
  srand(SDL_GetTicks());
}


/* Title screen: */

int title(void)
{
  int done, quit, i, x, y;
  SDL_Event event;
  SDL_Rect dest;

  
  /* Start music, if it's not playing: */
  
#ifndef NOSOUND
  if (use_sound)
    {
      if (!Mix_PlayingMusic())
	{
	  Mix_PlayMusic(mus_title, 0);
	}
    }
#endif
  
  
  /* Animate introduction of title screen: */
  
  for (i = 0; i <= 320; i = i + 32)
    {
      for (y = 240 - i; y < 240 + i; y = y + 16)
	{
	  for (x = 320 - i; x < 320 + i; x = x + 16)
	    {
	      dest.x = x;
	      dest.y = y;
	      dest.w = 16;
	      dest.h = 16;
	      
	      SDL_BlitSurface(images[IMG_TITLE], &dest,
			      screen, &dest);
	    }
	}
      
      SDL_Flip(screen);
      SDL_Delay(30);
    }
  
  
  /* Draw score: */
  
  drawdigits(score, 466, 422);
  drawdigits(level, 600, 422);
  
  
  /* Draw high score: */
  
  drawdigits(highscore, 466, 460);
  drawdigits(highlevel, 600, 460);
  

  SDL_Flip(screen);
  
  
  /* MAIN TITLE LOOP: */
  
  done = 0;
  quit = 0;
  
  do
    {
      while (SDL_PollEvent(&event) > 0)
	{
	  if (event.type == SDL_QUIT)
	    {
	      /* WM Quit - Quit! */
	      
	      quit = 1;
	    }
	  else if (event.type == SDL_KEYDOWN)
	    {
	      /* Keypress! ESCAPE - Quit, Space/Return - Start!*/
	      
	      if (event.key.keysym.sym == SDLK_ESCAPE)
		quit = 1;
	      else if (event.key.keysym.sym == SDLK_SPACE ||
		       event.key.keysym.sym == SDLK_RETURN)
		done = 1;
	    }
	  else if (event.type == SDL_MOUSEBUTTONDOWN)
	    {
	      /* Mouse click! - Start! */
	      
	      done = 1;
	    }
	}
      
      
      /* Keep playing music: */
      
#ifndef NOSOUND
      if (use_sound)
	{
	  if (!Mix_PlayingMusic())
	    {
	      Mix_PlayMusic(mus_title, 0);
	    }
	}
#endif
      
      
      /* Pause for a moment: */
      
      SDL_Delay(100);
    }
  while (!done && !quit);

  
  /* Silence music: */
  
#ifndef NOSOUND
  if (use_sound)
    Mix_HaltMusic();
#endif

  
  return(quit);
}


/* Set icon: */

void seticon(void)
{
  int masklen;
  Uint8 * mask;
  SDL_Surface * icon;
  
  
  /* Load icon into a surface: */
  
  icon = SDL_LoadBMP(datafilename("images/icon.bmp"));
  if (icon == NULL)
    {
      fprintf(stderr,
              "\nError: I could not load the icon image: %s\n"
              "The Simple DirectMedia error that occured was:\n"
              "%s\n\n", datafilename("images/icon.bmp"), SDL_GetError());
      exit(1);
    }
  
  
  /* Create mask: */
  
  masklen = (((icon -> w) + 7) / 8) * (icon -> h);
  mask = malloc(masklen * sizeof(Uint8));
  memset(mask, 0xFF, masklen);
  
  
  /* Set icon: */
  
  SDL_WM_SetIcon(icon, mask);
  
  
  /* Free icon surface & mask: */
  
  free(mask);
  SDL_FreeSurface(icon);
}


/* Pause screen: */

int pause_screen(void)
{
  SDL_Rect dest;
  SDL_Event event;
  int done, quit;
  
  
  /* Draw "Paused" notice: */
  
  dest.x = 192;
  dest.y = 208;
  dest.w = 256;
  dest.h = 64;
  
  SDL_BlitSurface(images[IMG_PAUSED], NULL, screen, &dest);
  SDL_UpdateRect(screen, dest.x, dest.y, dest.w, dest.h);
  
  
  /* Wait for space to be pressed again: */
  
  done = 0;
  quit = 0;
  
  do
    {
      while (SDL_PollEvent(&event) > 0)
	{
	  if (event.type == SDL_QUIT)
	    {
	      /* WM Quit - Quit! */
	      
	      quit = 1;
	    }
	  else if (event.type == SDL_KEYDOWN)
	    {
	      /* Keypress! ESCAPE - Quit, Space/Return - Continue!*/
	      
	      if (event.key.keysym.sym == SDLK_ESCAPE)
		quit = 1;
	      else if (event.key.keysym.sym == SDLK_SPACE ||
		       event.key.keysym.sym == SDLK_RETURN)
		done = 1;
	    }
	}
      
      
      /* Keep playing music: */
      
#ifndef NOSOUND
      if (use_sound)
	{
	  if (!Mix_PlayingMusic())
	    {
	      Mix_PlayMusic(mus_game, 0);
	    }
	}
#endif
      
      
      /* Pause for a moment: */
      
      SDL_Delay(100);
    }
  while (!done && !quit);
  
  return(quit);
}


/* Play a sound: */

void playsound(int snd, int chan)
{
#ifndef NOSOUND
  if (use_sound)
    Mix_PlayChannel(chan, sounds[snd], 0);
#endif
}


/* Start a new level: */

void resetlevel(void)
{
  int i;
  
  
  /* Reset all bugs: */
  
  for (i = 0; i < MAX_BUGS; i++)
    {
      bugs[i].alive = 0;
    }  
  
  
  /* Reset all splats: */
  
  for (i = 0; i < MAX_SPLATS; i++)
    {
      splats[i].active = 0;
    }  
  
  
  /* Add some: */
  
  bugs_added = 0;
  
  for (i = 0; i < level / 2 && i < MAX_BUGS_TO_START; i++)
    addbug();
  
  
  multiplier = 1;
}


/* Add a bug: */

void addbug(void)
{
  int i, found;
  
  
  /* Find a free slot: */
  
  found = -1;
  
  for (i = 0; i < MAX_BUGS && found == -1; i++)
    {
      if (bugs[i].alive == 0)
	found = i;
    }
  
  
  /* Add the bug: */
  
  if (found != -1)
    {
      bugs[found].alive = 1;
      
      bugs[found].kind = ((rand() % ((level / 3) + 1)) % NUM_BUGS);
      bugs[found].x = (rand() % 608);
      bugs[found].y = -30;
      bugs[found].xm = (rand() % (level * 2)) - level;
      bugs[found].xmm = (rand() % 5) - 2;
      bugs[found].ym = (rand() % level) + 2 + level;
      
      if ((rand() % CHANCE_OF_FAST_BUG) == 0)
	bugs[found].ym = bugs[found].ym + 16;
      
      if (bugs[found].ym > 24)
	bugs[found].ym = 24;
      
      bugs[found].target_y = ((480 - images[IMG_ARM1] -> h) +
			      (rand() % ((images[IMG_ARM1] -> h) / 2)));
      
      bugs_added++;
    }
}


/* Add a splat: */

void addsplat(int x, int y, int img)
{
  int i, found;
  
  
  /* Find a free slot: */
  
  found = -1;
  
  for (i = 0; i < MAX_SPLATS && found == -1; i++)
    {
      if (splats[i].active == 0)
	found = i;
    }
  
  
  /* Add the splat: */
  
  if (found != -1)
    {
      splats[found].active = 1;
      splats[found].img = img;
      splats[found].x = x;
      splats[found].y = y;
    }
}


/* Intro screen: */

void intro(void)
{
  float scale, scalespeed;
  int x, y, done, count;
  SDL_Rect src, dest;
  SDL_Event event;

  
  /* Start music! */

#ifndef NOSOUND
  if (use_sound)
    Mix_PlayMusic(mus_title, 0);
#endif

  
  /* Blank screen: */
  
  SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 0, 0, 0));
  
  
  /* Zoom it! */
  
  scalespeed = 0.1;
  done = 0;
  
  for (scale = 1; scale < 5 && done == 0; scale = scale + scalespeed)
    {
      scalespeed = scalespeed + 0.1;
      
      /* Draw each row and column... */
      
      for (y = 0; y < 96; y++)
	{
	  for (x = 0; x < 128; x++)
	    {
	      /* Blit one pixel: */
	      
	      src.x = x;
	      src.y = y;
	      src.w = 1;
	      src.h = 1;
	      
	      dest.x = 0;
	      dest.y = 0;
	      dest.w = 1;
	      dest.h = 1;
	      
	      SDL_BlitSurface(images[IMG_PRESENTS], &src,
			      screen, &dest);
	      
	      
	      /* Duplicate it as a rectangle: */
	      
	      dest.x = 320 - (64 * scale) + x * scale;
	      dest.y = 240 - (48 * scale) + y * scale;
	      dest.w = scale + 1;
	      dest.h = scale + 1;
	      
	      SDL_FillRect(screen, &dest,
			   * (Uint32 *) (screen->pixels));
	    }
	}
      
      
      /* Update screen and keep animation slow enough to see! */
      
      SDL_Flip(screen);
      SDL_Delay(100);

      while (SDL_PollEvent(&event))
	{
	  if (event.type == SDL_MOUSEBUTTONDOWN ||
	      event.type == SDL_KEYDOWN)
	    done = 1;
	}
    }
  
  
  /* Pause for a moment before erasing intro. screen... */
  
  count = 0;
  
  do
    {
      /* A keypress or mouse click will quit the intro. early: */
      
      while (SDL_PollEvent(&event))
	{
	  if (event.type == SDL_MOUSEBUTTONDOWN ||
	      event.type == SDL_KEYDOWN)
	    done = 1;
	}
      
      
      /* Wait for 1 second: */
      
      SDL_Delay(50);
      count++;
    }
  while (!done && count < 20);
}


/* Draw bonus numbers: */

void drawbonusnumbers(int num)
{
  int place, i, digit, nonzero, x;
  SDL_Rect dest;
  
  place = 10000;
  nonzero = 0;
  x = 286;
  
  for (i = 0; i < 5; i++)
    {
      digit = num / place;
      
      if (digit != 0)
	nonzero = 1;
      
      if (digit != 0 || nonzero)
	{
	  dest.x = x;
	  dest.y = 240;
	  dest.w = images[IMG_LARGE0 + digit] -> w;
	  dest.h = images[IMG_LARGE0 + digit] -> h;
	  
	  SDL_BlitSurface(images[IMG_LARGE0 + digit], NULL,
			  screen, &dest);
	  
	  x = x + dest.w;
	}
      
      num = num - (digit * place);
      
      place = place / 10;
    }
}


/* Add score (handle bonus blood, too!) */

void incrementscore(int amt)
{
  score = score + amt;
  
  
  /* Did we pass a point spot? */
  
  if (((score - amt) / POINTS_FOR_BONUS_BLOOD) <
      (score / POINTS_FOR_BONUS_BLOOD))
    addblood();
  
  if (score > highscore)
    {
      highscore = score;
      highlevel = level;
      
      if (has_highscore == 0)
	{
	  has_highscore = 1;
	  playsound(SND_HIGHSCORE, 1);
	}
    }
}


/* Add blood */

void addblood(void)
{
  if (blood < (images[IMG_GAUGE_FULL] -> h))
    {
      blood = blood + 20;
      
      if (blood > (images[IMG_GAUGE_FULL] -> h))
	blood = (images[IMG_GAUGE_FULL] -> h);
      
      playsound(SND_AH, 1);
    }
  else
    playsound(SND_GLUG, 1);
}


/* Draw 16x16 font digits on the screen: */

void drawdigits(int v, int x, int y)
{
  SDL_Rect src, dest;
  char str[10];
  int i;
  
  sprintf(str, "%d", v);
  for (i = 0; i < strlen(str); i++)
    {
      dest.x = i * 16 + x;
      dest.y = y;
      dest.w = 16;
      dest.h = 16;
      
      src.x = (str[i] - '0') * 16;
      src.y = 0;
      src.w = 16;
      src.h = 16;
      
      SDL_BlitSurface(images[IMG_NUMBERS], &src, screen, &dest);
    }
}


/* Load data from data file: */

void loaddata(void)
{
  FILE * fi;
  char temp[128];
  
  
  /* Set defaults in case we can't load: */
  
  score = 0;
  level = 0;
  highscore = 1234;
  highlevel = 5;
  
  
  /* Try to open file: */
  
  fi = opendata("r");
  if (fi != NULL)
    {
      do
	{
	  fgets(temp, sizeof(temp), fi);
	  
	  if (!feof(fi))
	    {
	      temp[strlen(temp) - 1] = '\0';
	      
	      
              /* Parse each line: */
              
              if (strstr(temp, "highscore=") == temp)
                {
                  highscore = atoi(temp + 10);
                  
                  if (highscore == 0)
                    highscore = 1234;
                }
	      else if (strstr(temp, "highwave=") == temp)
                {
                  highlevel = atoi(temp + 9);
                  
                  if (highlevel == 0)
                    highlevel = 5;
                }
              else if (strstr(temp, "finalscore=") == temp)
                {
                  score = atoi(temp + 11);
                }
	      else if (strstr(temp, "finalwave=") == temp)
                {
                  level = atoi(temp + 10);
                }
	    }
	}
      while (!feof(fi));
      
      fclose(fi);
    }
}


/* Save data to data file: */

void savedata(void)
{
  FILE * fi;
  
  
  /* Try to open file: */
  
  fi = opendata("w");
  if (fi != NULL)
    {
      fprintf(fi, "# Bug Squish data file\n\n");
      
      fprintf(fi, "highscore=%d\n", highscore);
      fprintf(fi, "highwave=%d\n\n", highlevel);

      fprintf(fi, "finalscore=%d\n", score);
      fprintf(fi, "finalwave=%d\n\n", level);
      
      fprintf(fi, "# (File automatically created.)\n");
      
      fclose(fi);
    }
}


/* Open the data file: */

FILE * opendata(char * mode)
{
  char * filename, * home;
  FILE * fi;
  
  
#ifdef LINUX
  /* Get home directory (from $HOME variable)... if we can't determine it,
     use the current directory ("."): */
  
  if (getenv("HOME") != NULL)
    home = getenv("HOME");
  else
    home = ".";
  
  
  /* Create the buffer for the filename: */
  
  filename = (char *) malloc(sizeof(char) * (strlen(home) +
                                             strlen("/.madbomber") + 1));
  
  strcpy(filename, home);
  strcat(filename, "/.bugsquish");
#elif __BEOS__
	filename = datafilename("/bugsquish.dat");
#else
  filename = "bugsquish.dat";
#endif
  
  
  /* Try opening the file: */
  
  fi = fopen(filename, mode);
  
  if (fi == NULL)
    {
      fprintf(stderr, "\nWarning: I could not open the options file "); 
      
      if (strcmp(mode, "r") == 0)
        fprintf(stderr, "for read:");
      else if (strcmp(mode, "w") == 0)
        fprintf(stderr, "for write:");
      
     fprintf(stderr, "\n%s\n"
              "The error that occured was:\n"
              "%s\n\n", filename, strerror(errno));
    }
  
  
  return(fi);
}



/* Parse command line arguments: */

void getargs(int argc, char * argv[])
{
  int i;
  

  /* Set defaults: */
  
  use_sound = 1;
  use_fullscreen = 0;


  /* Grab args: */
  
  for (i = 1; i < argc; i++)
    {
      if (strcmp(argv[i], "--fullscreen") == 0 ||
          strcmp(argv[i], "-f") == 0)
        use_fullscreen = 1;
      else if (strcmp(argv[i], "--disable-sound") == 0 ||
               strcmp(argv[i], "--nosound") == 0 ||
               strcmp(argv[i], "-q") == 0)
        use_sound = 0;
      else if (strcmp(argv[i], "--help") == 0 ||
               strcmp(argv[i], "-h") == 0)
        {
          printf("Bug Squish\n"
                 "\n"
                 "Version " VERSION "\n"
                 "New Breed Software, 2000\n"
                 "\n"
                 "Programming: Bill Kendrick\n"
                 "Graphics: Bill Kendrick\n"
                 "Music: Sami I Saarnio and 'Kure4Kancer'\n"
                 "\n"
                 "Game controls:\n"
                 "  Mouse Movement   - Move fly swatter\n"
                 "  Any Mouse Button - Squish /\n"
                 "\n"
                 "Run with \"--usage\" for command-line options...\n"
                 "\n");
          
          exit(0);
        }
      else if (strcmp(argv[i], "--usage") == 0 ||
               strcmp(argv[i], "-u") == 0)
        usage(0);
      else
        usage(1);
    }
}


/* Display usage screen: */

void usage(int ret)
{
  FILE * fs;
  
  if (ret == 1)
    fs = stderr;
  else
    fs = stdout;
  
  fprintf(fs,
	  "\nUsage: bugsquish [--disable-sound] [--fullscreen] |\n"
	  "                   [--help (-h) | --usage (-u)]\n\n");
  fprintf(fs,
          "  --disable-sound - "
          "Disable sound and music. (Also \"--nosound\" or \"-q\")\n");
  fprintf(fs,
          "  --fullscreen    - "
          "Display in full screen instead of a window, if possible.\n"
          "                    (Also \"-f\")\n");
  fprintf(fs, "\n");
  
  exit(ret);
}

char* datafilename(const char* name)
{	
	strcpy(datafile,datapath);

	strcat(datafile,name);
	return datafile;	
}