// DGen/SDL v1.17+
// SDL interface :)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <SDL.h>
#include <SDL_audio.h>

#include "md.h"
#include "rc.h"
#include "rc-vars.h"
#include "pd.h"
#include "pd-defs.h"
#include "font.h"

// Bad hack- extern slot etc. from main.cpp so we can save/load states
extern int slot;
void md_save(md &megad);
void md_load(md &megad);

// Temp space
static char temp[256];

// Define externed variables
struct bmap mdscr;
unsigned char *mdpal = NULL;
struct sndinfo sndi;
char *pd_options = "fX:Y:";

// Define our personal variables :)
// Graphics
static SDL_Surface *screen = NULL;
static SDL_Color colors[64];
static int ysize = 0, fullscreen = 0, bytes_pixel = 0, pal_mode = 0;
static int x_scale = 1, y_scale = 1, xs, ys;
// Sound
static SDL_AudioSpec spec;
static int snd_rate, snd_segs, snd_16bit, snd_buf;
static volatile int snd_read = 0;
static Uint8 *playbuf = NULL;
// Messages
static volatile int sigalrm_happened = 0;

// Number of seconds to sustain messages
#define MESSAGE_LIFE 3

// Catch SIGALRM
static void sigalrm_handler(int)
{
  sigalrm_happened = 1;
}

// Document the -f switch
void pd_help()
{
  printf(
  "    -f              Attempt to run fullscreen.\n"
  "    -X scale        Scale the screen in the X direction.\n"
  "    -Y scale        Scale the screen in the Y direction.\n"
  );
}

// Handle the -f switch
void pd_option(char c, const char *)
{
  if(c == 'f') fullscreen = 1;
  if(c == 'X') x_scale = atoi(optarg);
  if(c == 'Y') y_scale = atoi(optarg);
}

// Initialize SDL, and the graphics
int pd_graphics_init(int want_sound, int want_pal)
{
  SDL_Color color;

  pal_mode = want_pal;

  /* Neither scale value may be 0 or negative */
  if(x_scale <= 0) x_scale = 1;
  if(y_scale <= 0) y_scale = 1;

  if(SDL_Init(want_sound? (SDL_INIT_VIDEO | SDL_INIT_AUDIO) : (SDL_INIT_VIDEO)))
    {
      fprintf(stderr, "sdl: Couldn't init SDL: %s!\n", SDL_GetError());
      return 0;
    }
  atexit(SDL_Quit);
  ysize = (want_pal? 240 : 224);

  // Set screen size vars
  xs = 320*x_scale; ys = ysize*y_scale;

  // Make a 320x224 or 320x240 display for the MegaDrive, with an extra 16 lines
  // for the message bar.
  if(!(screen =
       SDL_SetVideoMode(xs, ys + 16, 0,
       fullscreen? (SDL_HWPALETTE | SDL_HWSURFACE | SDL_FULLSCREEN) :
		   (SDL_HWPALETTE | SDL_HWSURFACE))))
    {
      fprintf(stderr, "sdl: Couldn't set %dx%d video mode: %s!",
	      xs, ys + 16, SDL_GetError());
      return 0;
    }
  // We don't need setuid priveledges anymore
  if(getuid() != geteuid())
    setuid(getuid());

  // Set the titlebar
  SDL_WM_SetCaption("DGen "VER, "dgen");
  // Hide the cursor
  SDL_ShowCursor(0);

  // If we're in 8 bit mode, set color 0xff to white for the text,
  // and make a palette buffer
  if(screen->format->BitsPerPixel == 8)
    {
      color.r = color.g = color.b = 0xff;
      SDL_SetColors(screen, &color, 0xff, 1);
      mdpal = (unsigned char*)malloc(256);
      if(!mdpal)
        {
	  fprintf(stderr, "sdl: Couldn't allocate palette!\n");
	  return 0;
	}
    }
  
  // Set bytes per pixel
  bytes_pixel = screen->format->BytesPerPixel;
  
  // Ignore events besides quit and keyboard
  SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);
  SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
  SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
  SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE);
#ifndef __BEOS__
  SDL_EventState(SDL_JOYMOTION, SDL_IGNORE);
  SDL_EventState(SDL_JOYBUTTONDOWN, SDL_IGNORE);
  SDL_EventState(SDL_JOYBUTTONUP, SDL_IGNORE);
#endif
  SDL_EventState(SDL_SYSWMEVENT, SDL_IGNORE);

  // Set up the MegaDrive screen
  bytes_pixel = screen->format->BytesPerPixel;
  mdscr.w = 320 + 16;
  mdscr.h = ysize + 16;
  mdscr.bpp = screen->format->BitsPerPixel;
  mdscr.pitch = mdscr.w * bytes_pixel;
  mdscr.data = (unsigned char*) malloc(mdscr.pitch * mdscr.h);
  if(!mdscr.data)
    {
      fprintf(stderr, "sdl: Couldn't allocate screen!\n");
      return 0;
    }

  // Set SIGALRM handler (used to clear messages after 3 seconds)
  signal(SIGALRM, sigalrm_handler);

  // And that's it! :D
  return 1;
}

// Update palette
void pd_graphics_palette_update()
{
  int i;
  for(i = 0; i < 64; ++i)
    {
      colors[i].r = mdpal[(i << 2)  ];
      colors[i].g = mdpal[(i << 2)+1];
      colors[i].b = mdpal[(i << 2)+2];
    }
  SDL_SetColors(screen, colors, 0, 64);
}

// Update screen
// This code is fairly transmittable to any linear display, just change p to
// point to your favorite raw framebuffer. ;) But planar buffers are a 
// completely different deal...
// Anyway, feel free to use it in your implementation. :)
void pd_graphics_update()
{
  static int f = 0;
  int i, j, k;
  unsigned char *p, *q;
  
  // If you need to do any sort of locking before writing to the buffer, do so
  // here.
  if(SDL_MUSTLOCK(screen))
    SDL_LockSurface(screen);
  
  // If SIGALRM was tripped, clear message
  if(sigalrm_happened)
    {
      sigalrm_happened = 0;
      pd_clear_message();
    }

  p = (unsigned char*)screen->pixels;
  // 2696 = 336 * 8 + 8. 336 should be the width of mdscr, so we move 8 pixels
  // down and 8 to the right to skip the messy border.
  q = (unsigned char*)mdscr.data + 2696 * bytes_pixel;

  for(i = 0; i < ysize; ++i)
    {
#ifdef ASM_CTV
      if(dgen_craptv) switch(dgen_craptv)
        {
	// Blur, by Dave
	case CTV_BLUR:
	  if(mdscr.bpp == 16) blur_bitmap_16(q, 319);
	  else if(mdscr.bpp == 15) blur_bitmap_15(q, 319);
	  break;
	// Scanline, by Phil
	case CTV_SCANLINE:
	  if((i & 1) && (mdscr.bpp == 16 || mdscr.bpp == 15))
	    test_ctv(q, 320);
	  break;
	// Interlace, a hacked form of Scanline by me :)
	case CTV_INTERLACE:
	  if((i & 1) ^ (++f & 1) && (mdscr.bpp == 16 || mdscr.bpp == 15))
	    test_ctv(q, 320);
	  break;
	default:
	  break;
	}
#endif // ASM_CTV
      if(x_scale == 1)
        {
	  if(y_scale == 1)
	    {
	      memcpy(p, q, 320 * bytes_pixel);
	      p += screen->pitch;
	    }
	  else
	    {
	      for(j = 0; j < y_scale; ++j)
	        {
		  memcpy(p, q, 320 * bytes_pixel);
		  p += screen->pitch;
		}
	    }
	}
      else
        {
	  /* Stretch the scanline out */
	  switch(bytes_pixel)
	    {
	    case 1:
	      {
	        unsigned char *pp = p, *qq = q;
	        for(j = 0; j < 320; ++j, ++qq)
	          for(k = 0; k < x_scale; ++k)
		    *(pp++) = *qq;
	        if(y_scale != 1)
	          for(pp = p, j = 1; j < y_scale; ++j)
		    {
		      p += screen->pitch;
		      memcpy(p, pp, xs);
		    }
	      }
	      break;
	    case 2:
	      {
	        short *pp = (short*)p, *qq = (short*)q;
	        for(j = 0; j < 320; ++j, ++qq)
	          for(k = 0; k < x_scale; ++k)
		    *(pp++) = *qq;
	        if(y_scale != 1)
	          for(pp = (short*)p, j = 1; j < y_scale; ++j)
		    {
		      p += screen->pitch;
		      memcpy(p, pp, xs*2);
		    }
	      }
	      break;
	    case 3:
	      /* FIXME */
	      break;
	    case 4:
	      {
	        long *pp = (long*)p, *qq = (long*)q;
	        for(j = 0; j < 320; ++j, ++qq)
	          for(k = 0; k < x_scale; ++k)
		    *(pp++) = *qq;
	        if(y_scale != 1)
	          for(pp = (long*)p, j = 1; j < y_scale; ++j)
		    {
		      p += screen->pitch;
		      memcpy(p, pp, xs*4);
		    }
	      }
	      break;
	    }
	  p += screen->pitch;
	}
      q += mdscr.pitch;
    }
  // Unlock when you're done!
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  // Update the screen
  SDL_UpdateRect(screen, 0, 0, xs, ys);
}
  
// Callback for sound
static void _snd_callback(void*, Uint8 *stream, int len)
{
  int i;
  // Slurp off the play buffer
  for(i = 0; i < len; ++i)
    {
      if(snd_read == snd_buf) snd_read = 0;
      stream[i] = playbuf[snd_read++];
    }
}

// Initialize the sound
int pd_sound_init(int &format, int &freq, int &segs)
{
  SDL_AudioSpec wanted;

  // Set the desired format
  wanted.freq = freq;
  wanted.format = format;
  wanted.channels = 2;
  wanted.samples = 512;
  wanted.callback = _snd_callback;
  wanted.userdata = NULL;

  // Open audio, and get the real spec
  if(SDL_OpenAudio(&wanted, &spec) < 0)
    {
      fprintf(stderr, "sdl: Couldn't open audio: %s!\n", SDL_GetError());
      return 0;
    }
  // Check everything
  if(spec.channels != 2)
    {
      fprintf(stderr, "sdl: Couldn't get stereo audio format!\n");
      goto snd_error;
    }
  if(spec.format == PD_SND_16)
    snd_16bit = 1, format = PD_SND_16;
  else if(spec.format == PD_SND_8)
    snd_16bit = 0, format = PD_SND_8;
  else
    {
      fprintf(stderr, "sdl: Couldn't get a supported audio format!\n");
      goto snd_error;
    }
  // Set things as they really are
  snd_rate = freq = spec.freq;
  sndi.len = spec.freq / (pal_mode? 50 : 60);
  if(segs <= 4) segs = snd_segs = 4;
  else if(segs <= 8) segs = snd_segs = 8;
  else if(segs <= 16) segs = snd_segs = 16;
  else segs = snd_segs = 32;

  // Calculate buffer size
  snd_buf = sndi.len * snd_segs * (snd_16bit? 4 : 2);
  --snd_segs;
  // Allocate play buffer
  if(!(sndi.l = (signed short*) malloc(sndi.len * sizeof(signed short))) ||
     !(sndi.r = (signed short*) malloc(sndi.len * sizeof(signed short))) ||
     !(playbuf = (Uint8*)malloc(snd_buf)))
    {
      fprintf(stderr, "sdl: Couldn't allocate sound buffers!\n");
      goto snd_error;
    }

  // It's all good!
  return 1;

snd_error:
  // Oops! Something bad happened, cleanup.
  SDL_CloseAudio();
  if(sndi.l) free((void*)sndi.l);
  if(sndi.r) free((void*)sndi.r);
  if(playbuf) free((void*)playbuf);
  return 0;
}

// Start/stop audio processing
void pd_sound_start()
{
  SDL_PauseAudio(0);
}

void pd_sound_pause()
{
  SDL_PauseAudio(1);
}

// Return segment we're currently playing from
int pd_sound_rp()
{
  return (snd_read / (sndi.len << (1+snd_16bit))) & snd_segs;
}

// Write contents of sndi to playbuf
void pd_sound_write(int seg)
{
  int i;
  signed short *w =
    (signed short*)(playbuf + seg * (sndi.len << (1+snd_16bit)));

  // Thanks Daniel Wislocki for this much improved audio loop :)
  if(!snd_16bit)
    {
      SDL_LockAudio();
      for(i = 0; i < sndi.len; ++i)
	*w++ = ((sndi.l[i] & 0xFF00) + 0x8000) | ((sndi.r[i] >> 8) + 0x80);
      SDL_UnlockAudio();
    } else {
      SDL_LockAudio();
      for(i = 0; i < sndi.len; ++i)
	{
	  *w++ = sndi.l[i];
	  *w++ = sndi.r[i];
	}
      SDL_UnlockAudio();
    }
}

// This is a small event loop to handle stuff when we're stopped.
static int stop_events(md &/*megad*/)
{
  SDL_Event event;

  // We still check key events, but we can wait for them
  while(SDL_WaitEvent(&event))
    {
      switch(event.type)
        {
	case SDL_KEYDOWN:
	  // We can still quit :)
	  if(event.key.keysym.sym == dgen_quit) return 0;
	  else return 1;
	case SDL_QUIT:
	  return 0;
	default: break;
	}
    }
  // SDL_WaitEvent only returns zero on error :(
  fprintf(stderr, "sdl: SDL_WaitEvent broke: %s!", SDL_GetError());
  return 1;
}

// The massive event handler!
// I know this is an ugly beast, but please don't be discouraged. If you need
// help, don't be afraid to ask me how something works. Basically, just handle
// all the event keys, or even ignore a few if they don't make sense for your
// interface.
int pd_handle_events(md &megad)
{
  SDL_Event event;

  // If there's any chance your implementation might run under Linux, add these
  // next four lines for joystick handling.
#ifdef JOYSTICK_SUPPORT
  if(dgen_joystick)
    megad.read_joysticks();
#endif
  // Check key events
  while(SDL_PollEvent(&event))
    {
      switch(event.type)
	{
	case SDL_KEYDOWN:
	  // Check if it was a significant key that was pressed
	  if(event.key.keysym.sym == pad1_up) megad.pad[0] &= ~0x01;
	  else if(event.key.keysym.sym == pad1_down) megad.pad[0] &= ~0x02;
	  else if(event.key.keysym.sym == pad1_left) megad.pad[0] &= ~0x04;
	  else if(event.key.keysym.sym == pad1_right) megad.pad[0] &= ~0x08;
	  else if(event.key.keysym.sym == pad1_a) megad.pad[0] &= ~0x1000;
	  else if(event.key.keysym.sym == pad1_b) megad.pad[0] &= ~0x10;
	  else if(event.key.keysym.sym == pad1_c) megad.pad[0] &= ~0x20;
	  else if(event.key.keysym.sym == pad1_x) megad.pad[0] &= ~0x40000;
	  else if(event.key.keysym.sym == pad1_y) megad.pad[0] &= ~0x20000;
	  else if(event.key.keysym.sym == pad1_z) megad.pad[0] &= ~0x10000;
	  else if(event.key.keysym.sym == pad1_mode) megad.pad[0] &= ~0x80000;
	  else if(event.key.keysym.sym == pad1_start) megad.pad[0] &= ~0x2000;

	  else if(event.key.keysym.sym == pad2_up) megad.pad[1] &= ~0x01;
	  else if(event.key.keysym.sym == pad2_down) megad.pad[1] &= ~0x02;
	  else if(event.key.keysym.sym == pad2_left) megad.pad[1] &= ~0x04;
	  else if(event.key.keysym.sym == pad2_right) megad.pad[1] &= ~0x08;
	  else if(event.key.keysym.sym == pad2_a) megad.pad[1] &= ~0x1000;
	  else if(event.key.keysym.sym == pad2_b) megad.pad[1] &= ~0x10;
	  else if(event.key.keysym.sym == pad2_c) megad.pad[1] &= ~0x20;
	  else if(event.key.keysym.sym == pad2_x) megad.pad[1] &= ~0x40000;
	  else if(event.key.keysym.sym == pad2_y) megad.pad[1] &= ~0x20000;
	  else if(event.key.keysym.sym == pad2_z) megad.pad[1] &= ~0x10000;
	  else if(event.key.keysym.sym == pad2_mode) megad.pad[1] &= ~0x80000;
	  else if(event.key.keysym.sym == pad2_start) megad.pad[1] &= ~0x2000;

	  else if(event.key.keysym.sym == dgen_quit) return 0;
// Split screen is unnecessary with new renderer.
//	  else if(event.key.keysym.sym == dgen_splitscreen_toggle)
//	    {
//	      split_screen = !split_screen;
//	      pd_message(split_screen? "Split screen enabled." : 
//				       "Split screen disabled.");
//	    }
	  else if(event.key.keysym.sym == dgen_craptv_toggle)
	    {
		  dgen_craptv = ((++dgen_craptv) % NUM_CTV);
		  sprintf(temp, "Crap TV mode \"%s\".", ctv_names[dgen_craptv]);
		  pd_message(temp);
	    }
	  else if(event.key.keysym.sym == dgen_reset)
	    { megad.reset(); pd_message("Genesis reset."); }
	  else if(event.key.keysym.sym == dgen_slot_0)
	    { slot = 0; pd_message("Selected save slot 0."); }
	  else if(event.key.keysym.sym == dgen_slot_1)
	    { slot = 1; pd_message("Selected save slot 1."); }
	  else if(event.key.keysym.sym == dgen_slot_2)
	    { slot = 2; pd_message("Selected save slot 2."); }
	  else if(event.key.keysym.sym == dgen_slot_3)
	    { slot = 3; pd_message("Selected save slot 3."); }
	  else if(event.key.keysym.sym == dgen_slot_4)
	    { slot = 4; pd_message("Selected save slot 4."); }
	  else if(event.key.keysym.sym == dgen_slot_5)
	    { slot = 5; pd_message("Selected save slot 5."); }
	  else if(event.key.keysym.sym == dgen_slot_6)
	    { slot = 6; pd_message("Selected save slot 6."); }
	  else if(event.key.keysym.sym == dgen_slot_7)
	    { slot = 7; pd_message("Selected save slot 7."); }
	  else if(event.key.keysym.sym == dgen_slot_8)
	    { slot = 8; pd_message("Selected save slot 8."); }
	  else if(event.key.keysym.sym == dgen_slot_9)
	    { slot = 9; pd_message("Selected save slot 9."); }
	  else if(event.key.keysym.sym == dgen_save) md_save(megad);
	  else if(event.key.keysym.sym == dgen_load) md_load(megad);
// Added this CPU core hot swap.  Compile both Musashi and StarScream
// in, and swap on the fly like DirectX DGen. [PKH]
#if defined (COMPILE_WITH_MUSA) && (COMPILE_WITH_STAR)
	  else if(event.key.keysym.sym == dgen_cpu_toggle)
	    {
	      if(megad.cpu_emu) {
	        megad.change_cpu_emu(0);
  	        pd_message("StarScream CPU core activated.");
              }
	      else {
	        megad.change_cpu_emu(1);
	        pd_message("Musashi CPU core activated."); 
              }
	    }
#endif
	  else if(event.key.keysym.sym == dgen_stop) {
	    pd_message("* STOPPED * Press any key to continue.");
	    SDL_PauseAudio(1); // Stop audio :)
	    int r = stop_events(megad);
	    pd_clear_message();
	    if(r) SDL_PauseAudio(0); // Restart audio
	    return r;
	  }
	  else if(event.key.keysym.sym == dgen_fix_checksum) {
	    pd_message("Checksum fixed.");
	    megad.fix_rom_checksum();
	  }
	  break;
	case SDL_KEYUP:
	  // The only time we care about key releases is for the controls
	  if(event.key.keysym.sym == pad1_up) megad.pad[0] |= 0x01;
	  else if(event.key.keysym.sym == pad1_down) megad.pad[0] |= 0x02;
	  else if(event.key.keysym.sym == pad1_left) megad.pad[0] |= 0x04;
	  else if(event.key.keysym.sym == pad1_right) megad.pad[0] |= 0x08;
	  else if(event.key.keysym.sym == pad1_a) megad.pad[0] |= 0x1000;
	  else if(event.key.keysym.sym == pad1_b) megad.pad[0] |= 0x10;
	  else if(event.key.keysym.sym == pad1_c) megad.pad[0] |= 0x20;
	  else if(event.key.keysym.sym == pad1_x) megad.pad[0] |= 0x40000;
	  else if(event.key.keysym.sym == pad1_y) megad.pad[0] |= 0x20000;
	  else if(event.key.keysym.sym == pad1_z) megad.pad[0] |= 0x10000;
	  else if(event.key.keysym.sym == pad1_mode) megad.pad[0] |= 0x80000;
	  else if(event.key.keysym.sym == pad1_start) megad.pad[0] |= 0x2000;

	  else if(event.key.keysym.sym == pad2_up) megad.pad[1] |= 0x01;
	  else if(event.key.keysym.sym == pad2_down) megad.pad[1] |= 0x02;
	  else if(event.key.keysym.sym == pad2_left) megad.pad[1] |= 0x04;
	  else if(event.key.keysym.sym == pad2_right) megad.pad[1] |= 0x08;
	  else if(event.key.keysym.sym == pad2_a) megad.pad[1] |= 0x1000;
	  else if(event.key.keysym.sym == pad2_b) megad.pad[1] |= 0x10;
	  else if(event.key.keysym.sym == pad2_c) megad.pad[1] |= 0x20;
	  else if(event.key.keysym.sym == pad2_x) megad.pad[1] |= 0x40000;
	  else if(event.key.keysym.sym == pad2_y) megad.pad[1] |= 0x20000;
	  else if(event.key.keysym.sym == pad2_z) megad.pad[1] |= 0x10000;
	  else if(event.key.keysym.sym == pad2_mode) megad.pad[1] |= 0x80000;
	  else if(event.key.keysym.sym == pad2_start) megad.pad[1] |= 0x2000;
	  break;
	case SDL_QUIT:
	  // We've been politely asked to exit, so let's leave
	  return 0;
	default:
	  break;
	}
    }
  return 1;
}

// Write a message to the status bar
void pd_message(const char *msg)
{
  pd_clear_message();
  font_text(screen, 0, ys, msg);
  SDL_UpdateRect(screen, 0, ys, xs, 16);
  // Clear message in 3 seconds
  alarm(MESSAGE_LIFE);
}

inline void pd_clear_message()
{
  int i, j;
  long *p = (long*)((char*)screen->pixels + (screen->pitch * ys));
  for(i = 0; i < 16; ++i, p += (screen->pitch >> 2))
    for(j = 0; j < 80 * screen->format->BytesPerPixel; ++j)
      p[j] = 0;
  SDL_UpdateRect(screen, 0, ys, xs, 16);
}

/* FIXME: Implement this
 * Look at v1.16 to see how I did carthead there */
void pd_show_carthead(md&)
{
}

/* Clean up this awful mess :) */
void pd_quit()
{
  if(mdscr.data) free((void*)mdscr.data);
  if(playbuf)
    {
      SDL_CloseAudio();
      free((void*)playbuf);
    }
  if(sndi.l) free((void*)sndi.l);
  if(sndi.r) free((void*)sndi.r);
  if(mdpal) free((void*)mdpal);
}
